unit IWCompOrderedListbox;

interface
uses
  {$IFDEF VSNET}
  IWNetComponent,
  System.ComponentModel, System.Drawing,
  IWNetClasses,
  {$ELSE}
  {$IFDEF Linux}IWCLXClasses,{$ELSE}IWVCLClasses,{$ENDIF}
  {$ENDIF}
  Classes,
  IWBaseInterfaces, IWCompButton, IWCompListbox, IWControl,
  IWGrids, IWRenderContext, IWHTMLTag, IWFont;

type
  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWOrderedListbox.bmp}
  TIWOrderedListbox = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWOrderedListbox), 'TIWOrderedListbox.bmp')]
  {$ENDIF}
  TIWOrderedListbox = class(TIWCustomControl, IIWSubmitControl, IIWInputControl)
  private
    FSubmitParam : string;

    // Internal
    FButtonWidth : integer;
    FItems : TIWStringList;

    // Properties
    FMoveUpCaption,
    FMoveDownCaption : string;

    FUpID,
    FDownID,
    FListID : string;

    // fonts for visual elements
    FItemsFont : TIWFont;
    FCaptionFont : TIWFont;
    FButtonsFont : TIWFont;
    //
    function GetItems: TIWStringList;
    procedure SetItems(const Value: TIWStringList);

  protected
    FItemIndex: Integer;
    FMultiSelect: Boolean;
    FSelectedList: TStringList;

    procedure InitControl; override;
    procedure InitDesignTime; override;
    procedure Submit(const AValue: string); override;

    procedure GetInputControlNames(ANames: TStringList); override;
    function IsForThisControl(AName: string): boolean; override;
    procedure SetValue(const AValue: string); virtual;

    {$IFDEF CLR}strict protected{$ELSE}protected{$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
    procedure SetItemIndex(AIndex: Integer); virtual;
    procedure SetMultiSelect(const AValue: Boolean);
    function GetSelected(AIndex: integer): boolean;
    procedure SetSelected(AIndex: integer; const AValue: boolean);
    procedure ResetSelection;
  public
    function GetSubmitParam : string;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    property Selected[AIndex: Integer]: Boolean read GetSelected write SetSelected;
  published
    property ButtonWidth : integer read FButtonWidth write FButtonWidth;
    property Caption;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Items : TIWStringList read GetItems write SetItems;
    property ItemIndex: Integer read FItemIndex write SetItemIndex;
    property MultiSelect: boolean read FMultiSelect write SetMultiSelect;
    property MoveDownCaption : string read FMoveDownCaption write FMoveDownCaption;
    property MoveUpCaption : string read FMoveUpCaption write FMoveUpCaption;
    property ItemsFont : TIWFont read FItemsFont write FItemsFont;
    property CaptionFont : TIWFont read FCaptionFont write FCaptionFont;
    property ButtonsFont : TIWFont read FButtonsFont write FButtonsFont;
  end;

implementation
uses
  IWBaseRenderContext, IWContainerLayout, IWHTML40Interfaces,
  IWResourceStrings, IWMarkupLanguageTag, IWServer,
  Math,
  SWStrings, SWSystem, SysUtils;

{ TIWOrderedListbox }

procedure TIWOrderedListbox.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FItems);
  FreeAndNil(FSelectedList);
  FreeAndNil(FItemsFont);
  FreeAndNil(FCaptionFont);
  FreeAndNil(FButtonsFont);
  inherited Dispose(ADispose);
end;

function TIWOrderedListbox.GetItems: TIWStringList;
begin
  Result := FItems;
end;

function TIWOrderedListbox.GetSubmitParam: string;
begin
  Result := FSubmitParam;
end;

procedure TIWOrderedListbox.InitControl;
begin
  inherited InitControl;

  FItems := TIWStringList.Create;
  FSelectedList := TStringList.Create;

  Caption := 'Use the Move up and Move down buttons to sort items.';

  FButtonWidth := 0;
  FMoveDownCaption := 'Move down';
  FMoveUpCaption := 'Move up';

  FItemsFont := IWFont.TIWFont.Create;
  FCaptionFont := IWFont.TIWFont.Create;
  FButtonsFont := IWFont.TIWFont.Create;
end;

procedure TIWOrderedListbox.InitDesignTime;
begin
  inherited InitDesignTime;
end;

function TIWOrderedListbox.RenderHTML(
  AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  f : integer;
begin
  // Setup internal HTML control names
  FUpID := HTMLName + 'up';
  FDownID := HTMLName + 'down';
  FListID := HTMLName + 'list';

  // Load the required javascript functions
  (AContext as TIWComponent40Context).AddScriptFile('/js/IWOrderedListbox.js');

  //
  Result := TIWHTMLTag.CreateTag('TABLE');
  try
    Result.AddStringParam('ID', HTMLName);

    Result.Contents.AddText(iif(Caption, '<TR><TD COLSPAN=2><CAPTION ' +
                                         iif((FCaptionFont.CSSStyle <> ''),
                                          'CLASS = "' + FCaptionFont.CSSStyle + '"',
                                          'STYLE = "' + FCaptionFont.FontToStringStyle(AContext.Browser) + '"')
                                         + '>' + Caption + '</CAPTION></TD></TR>'));

    with Result.Contents do
    begin
      // Add visual stuff
      with AddTag('TR').Contents do
      begin
        // Add the selection box
        with AddTag('TD') do
        begin
          AddStringParam('ROWSPAN', '2');
          AddStringParam('VALIGN', 'MIDDLE');
          AddStringParam('ALIGN', 'CENTER');
          with Contents.AddTag('SELECT') do
          begin
            Add(iif(FMultiSelect, 'MULTIPLE'));
            AddIntegerParam('SIZE', Height div 16);
            if FItemsFont.CSSStyle <> '' then
              begin
                AddStringParam('STYLE', 'width:' + IntToStr(iif(FButtonWidth <> 0, Width - FButtonWidth - 16, 2 * Width div 3 - 16)) + ';');
                AddStringParam('CLASS', FItemsFont.CSSStyle);
              end
            else
              begin
                AddStringParam('STYLE', FItemsFont.FontToStringStyle(AContext.Browser) +
                 'width:' + IntToStr(iif(FButtonWidth <> 0, Width - FButtonWidth - 16, 2 * Width div 3 - 16)) + ';');
              end;
            AddStringParam('ID', FListID);
            AddStringParam('OnChange', 'DoOrderedListClick(this, ''' + FUpID + ''', ''' + FDownID + ''', ''' + HTMLName + '_VALUE'')');
            with Contents do
            begin
              for f := 0 to Pred(FItems.Count) do
              begin
                with AddTag('OPTION') do
                begin
                  AddStringParam('VALUE', FItems.ValueFromIndex[f]);
                  if (FMultiSelect and Selected[f]) or (not FMultiSelect and (FItemIndex = f)) then
                  begin
                    Add('SELECTED');
                  end;
                  Contents.AddText(FItems.Names[f]);
                end;
              end;
            end;
          end;
        end;
        // Add the move up button
        with AddTag('TD') do
        begin
          AddStringParam('VALIGN', 'BOTTOM');
          AddStringParam('ALIGN', 'CENTER');
          with Contents.AddTag('INPUT') do
          begin
            AddStringParam('ID', FUpID);
            AddStringParam('TYPE', 'BUTTON');
            AddStringParam('VALUE', FMoveUpCaption);
            AddStringParam('DISABLED', 'true');
            if (FButtonsFont.CSSStyle <> '') then
              begin
                AddStringParam('STYLE', 'width:' + IntToStr(iif(FButtonWidth <> 0, FButtonWidth, Width div 3)));
                AddStringParam('CLASS', FButtonsFont.CSSStyle);
              end
            else
              begin
                AddStringParam('STYLE', FButtonsFont.FontToStringStyle(AContext.Browser) +
                    'width:' + IntToStr(iif(FButtonWidth <> 0, FButtonWidth, Width div 3)));
              end;
            AddStringParam('onclick', 'MoveOrderedListItem(1, ''' +
              FListID + ''', ''' + FUpID + ''', ''' + FDownID + ''', ''' + HTMLName + '_VALUE'')');
          end;
        end;
      end;
      with AddTag('TR').Contents do
      begin
        // Add the move down button
        with AddTag('TD') do
        begin
          AddStringParam('VALIGN', 'TOP');
          AddStringParam('ALIGN', 'CENTER');
          with Contents.AddTag('INPUT') do
          begin
            AddStringParam('ID', FDownID);
            AddStringParam('TYPE', 'BUTTON');
            AddStringParam('VALUE', FMoveDownCaption);
            if (FButtonsFont.CSSStyle <> '') then
              begin
                AddStringParam('STYLE', 'width:' + IntToStr(iif(FButtonWidth <> 0, FButtonWidth, Width div 3)));
                AddStringParam('CLASS', FButtonsFont.CSSStyle);
              end
            else
              begin
                AddStringParam('STYLE', FButtonsFont.FontToStringStyle(AContext.Browser) +  
                    'width:' + IntToStr(iif(FButtonWidth <> 0, FButtonWidth, Width div 3)));
              end;
            AddStringParam('onclick', 'MoveOrderedListItem(-1, ''' +
              FListID + ''', ''' + FUpID + ''', ''' + FDownID + ''', ''' + HTMLName + '_VALUE'')');
          end;
        end;
      end;
      // Add selection field
      with AddTag('INPUT') do
      begin
        AddStringParam('TYPE', 'HIDDEN');
        AddStringParam('NAME', HTMLName + '_VALUE');
      end;
    end;
  except
    FreeAndNil(Result);
    raise;
  end;
end;

procedure TIWOrderedListbox.SetItems(const Value: TIWStringList);
begin
  FItems.Assign(Value);
end;

procedure TIWOrderedListbox.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
end;

procedure TIWOrderedListbox.GetInputControlNames(ANames: TStringList);
begin
  ANames.Add(HTMLName + '_VALUE');
end;

function TIWOrderedListbox.IsForThisControl(AName: string): boolean;
begin
  Result := HTMLName + '_VALUE' = AName;
end;

procedure TIWOrderedListbox.SetValue(const AValue: string);
var
  LTempList : TStringList;
  g : integer;
  s, t : string;
  LSelection : boolean;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  if AValue <> '' then
  begin
    t := AValue;
    LTempList := TStringList.Create;
    ResetSelection;
    try
      s := Fetch(t, ';');
      while s <> '' do
      begin
        LSelection := False;
        if s[1] = '*' then
        begin
          s := Copy(s, 2, Length(s) - 1);
          LSelection := True;
        end;
        for g := 0 to Pred(FItems.Count) do
        begin
          if FItems.ValueFromIndex[g] = s then
          begin
            LTempList.Add(FItems[g]);
            if LSelection then
            begin
              if FMultiSelect then
              begin
                FSelectedList.Add(FItems[g])
              end
              else
              begin
                FItemIndex := Pred(LTempList.Count);
              end;
            end;
            Break;
          end;
        end;
        s := Fetch(t, ';');
      end;
      FItems.Assign(LTempList);
    finally
      FreeAndNil(LTempList);
    end;
  end;
end;

procedure TIWOrderedListbox.SetItemIndex(AIndex: Integer);
begin
  if IsLoading then begin
    // Set no matter what, it might be set (and usually is) before the items are loaded
    FItemIndex := AIndex;
  end
  else
  begin
    if AIndex < Items.Count then
    begin
      if FItemIndex <> AIndex then
      begin
        FItemIndex := AIndex;
        DoRefreshControl := True;
        Invalidate;
      end;
    end;
  end;
end;

procedure TIWOrderedListbox.SetMultiSelect(const AValue: boolean);
begin
  FMultiSelect := AValue;
end;

function TIWOrderedListbox.GetSelected(AIndex: Integer): boolean;
begin
  if FMultiSelect then
  begin
    Result := FSelectedList.IndexOf(FItems[AIndex]) > -1;
  end
  else
  begin
    Result := AIndex = ItemIndex;
  end;
end;

procedure TIWOrderedListbox.SetSelected(AIndex: integer; const AValue: boolean);
begin
  if AValue then
  begin
    if not GetSelected(AIndex) then
    begin
      FSelectedList.Add(FItems[AIndex]);
    end;
  end
  else
  begin
    FSelectedList.Delete(FSelectedList.IndexOf(FItems[AIndex]));
  end;
  Invalidate;
end;

procedure TIWOrderedListbox.ResetSelection;
begin
  FSelectedList.Clear;
  FItemIndex := -1;
end;

initialization
  TIWServer.AddInternalFile('IW_JS_IWORDEREDLISTBOX', '/js/IWOrderedListbox.js');

end.
